// Building up towards LISP 1.5 in Javascript

// START: preamble
lodash = require('lodash')
util = require('util')

debug = false
log_debug = (...args) => debug && console.log(...args)
// END: preamble

console.log('')
console.log('==========================================================')
console.log('')

// simple helper for assertions.
// START: assert
testCount = 0
function assert(a) {
  testCount++
  if (!a)
    throw new Error("Assertion failed")
}
function assertEqual(a, b) {
  testCount++
  if (!lodash.isEqual(a, b))
    throw new Error(
        'Assertion failed, not equals: ' +
        `${util.inspect(a)} != ${util.inspect(b)}`)
}
// END: assert

// The most elemntary function of them all
// START: cons
cons = (a, b) => [a, b]

assertEqual(cons('A', 'B'),
            ['A', 'B'])
assertEqual(cons(cons('A', 'B'), 'C'),
            [['A', 'B'], 'C'])
// END: cons

// START: carcdr
car = l => Array.isArray(l) ? l[0] : undefined

assertEqual(car(['A', 'B']),
            'A')
assertEqual(car(['A', ['B1', 'B2']]),
            'A')
assertEqual(car([['A1', 'A2'], 'B']),
            ['A1', 'A2'])
assertEqual(car('A'),
            undefined)

cdr = l => Array.isArray(l) ? l[1] : undefined

assertEqual(cdr(['A', 'B']),
            'B')
assertEqual(cdr(['A', ['B1', 'B2']]),
            ['B1', 'B2'])
assertEqual(cdr([['A1', 'A2'], 'B']),
            'B')
assertEqual(cdr('A'),
            undefined)
assertEqual(car(cdr(['A', ['B1', 'B2']])),
            'B1')
assertEqual(car(cdr(['A', 'B'])),
            undefined)
assertEqual(car(cons('A', 'B')),
            'A')
// END: carcdr

// We move the definition of atom up so we can
// use it in eq. Note that 'true', 'false' and 'null'
// are atoms that stand for 'T', 'F' and 'null'
// START: atom
atom = (a) => {
  return a === null ||
    a === true ||
    a === false ||
    (typeof(a) === 'string' && /^[A-Z][A-Z0-9]*$/.test(a))
}

assert(atom('EXTRALONGSTRINGOFLETTERS'))
assert(!atom(cons('U', 'V')))
assert(atom(car(['U', 'V'])))
// END: atom

// START: eq
eq = (a, b) => {
  let retval = null
  if (!(atom(a) && atom(b)))
    retval = undefined
  else
    retval = (a == b)
  debug && log_debug("eq?", a, b, ':', retval)
  return retval
}

assert(eq('A', 'A'))
assert(!eq(cons('A', 'B'), cons('A', 'B')))

assertEqual(eq('A', ['A', 'B']),
            undefined)
assertEqual(eq(['A', 'B'], ['A', 'B']),
            undefined)
// END: eq

// helper to construct lists so we don't need to write out nested conses. This
// one is not part of the Lisp definition.
// START: list
list = (l) => l.length == 0 ?
  null : cons(l[0], list(l.slice(1)))

assertEqual(list('A', 'B', 'C'),
            ['A', ['B', ['C', null]]])
assertEqual(list(list('A', 'B'), 'C'),
            [['A', ['B', null]], ['C', null]])
assertEqual(list('A', 'B', list('C', 'D')),
            ['A', ['B', [['C', ['D', null]], null]]])
assertEqual(list('A'),
            ['A', null])
assertEqual(list(list('A')),
            [['A', null], null])
assertEqual(list('A', cons('B', 'C')),
            ['A', [['B', 'C'], null]])
assertEqual(car(list('A', 'B', 'C')),
            'A')
assertEqual(cdr(list('A', 'B', 'C')),
            list('B', 'C'))
assertEqual(cons('A', list('B', 'C')),
            list('A', 'B', 'C'))
assertEqual(car(list(list('A', 'B'), 'C')),
            list('A', 'B'))
assertEqual(car(cdr(list('A', 'B', 'C'))),
            'B')
// END: list

cadr = l => car(cdr(l))
caddr = l => car(cdr(cdr(l)))
cadadr = l => car(cdr(car(cdr(l))))

assertEqual(cadr(list('A', 'B', 'C')),
            'B')
assertEqual(caddr(list('A', 'B', 'C')),
            'C')
assertEqual(cadadr(list('A', list('B', 'C'), 'D')),
            'C')

// For conditionals, we use the ternary operator as a first
// approximation of `cond`. Not perfect but good enough to
// be workable.
//
// A conditional expression has the following form:
// [p1 -> e1; p2 -> e2; ... ; pn -> en]
// we translate that to
// p1 ? e1 : p2 ? e2 : .... : pn ? en : undefined
// Which is pretty much as short except for having to end with 'undefined'
// we allow one shorthand: "T -> x" as the final clause should be
// mechanically translated as ": T ? x : undefined" but is a bit
// simpler to read if we write it as ": x"

// START: ff
ff = x => atom(x) ? x : true ? ff(car(x)) : undefined

assertEqual(ff(list('A', 'B', 'C')), 'A')
// END: ff


// A Universal Lisp Function

// START: equal
equal = (x, y) => atom(x) ? (atom(y) ? eq(x, y) : false)
  : equal(car(x), car(y)) ? equal(cdr(x), cdr(y)) : false

assert(equal('A', 'A'))
assert(equal(list('A', 'B'),
             list('A', 'B')))
assert(!equal('A',
              null))
assert(!equal('A',
              list('A', 'B')))
// END: equal

// START: subst
subst = (x, y, z) => equal(x, z) ? x :
  atom(z) ? z :
  cons(subst(x, y, car(z)), subst(x, y, cdr(z)))

assert(subst(cons('X', 'A'), 'B', cons(cons('A', 'B'), 'C')),
       cons(cons('A', cons('X', 'A')), 'C'))
// END: subst

// We can't use 'null' so we use nil
// START: nil
nil = x => eq(x, null)

assert(nil(null))
assert(nil(cdr(list('A'))))
// END: nil

// START: append
append = (x, y) => nil(x) ? y : cons(car(x), append(cdr(x), y))

assertEqual(append(list('A', 'B'), list('C', 'D', 'E')),
            list('A', 'B', 'C', 'D', 'E'))
// END: append

// START: member
member = (x, y) => nil(y) ? false : equal(x, car(y)) ?
  true :
  member(x, cdr(y))

assert(member('A', list('B', 'C', 'A', 'D')))
assert(!member('F', list('B', 'C', 'A', 'D')))
// END: member

// START: pairlis
pairlis = (x, y, a) => nil(x) ? a :
  cons(cons(car(x), car(y)), pairlis(cdr(x), cdr(y), a))

assertEqual(pairlis(list('A', 'B', 'C'), list('U', 'V', 'W'),
                    list(cons('D', 'X'), cons('E', 'Y'))),
            list(cons('A', 'U'), cons('B', 'V'), cons('C', 'W'),
                 cons('D', 'X'), cons('E', 'Y')))
// END: pairlis

// START: assoc
caar = l => car(car(l))
assoc = (x, a) => equal(caar(a), x) ? car(a) : assoc(x, cdr(a))

assertEqual(assoc('B', list(cons('A', list('M', 'N')),
                            cons('B', list('CAR', 'X')),
                            cons('C', list('QUOTE', 'M')),
                            cons('C', list('CDR', 'X')))),
            cons('B', list('CAR', 'X')))
// END: assoc


// START: sublis
cdar = l => cdr(car(l))
sub2 = (a, z) => nil(a) ? z :
  eq(caar(a), z) ? cdar(a) :
  sub2(cdr(a), z)
sublis = (a, y) => atom(y) ? sub2(a, y) :
  cons(sublis(a, car(y)), sublis(a, cdr(y)))

assertEqual(sublis(list(cons('X', 'SHAKESPEARE'),
                        cons('Y', list('THE', 'TEMPEST'))),
                   list('X', 'WROTE', 'Y')),
           list('SHAKESPEARE', 'WROTE', list('THE', 'TEMPEST')))
// END: sublis

// Ok, now for the meat - apply and eval. We rename them as `eval` is already part of JS

// START: apply
lisp_apply = (fn, x, a) => {
  console.log('apply >', pp(fn), pp(x), pp(a))
  let retval =
    atom(fn) ? (
      eq(fn, 'CAR') ? caar(x) :
      eq(fn, 'CDR') ? cdar(x) :
      eq(fn, 'CONS') ? cons(car(x), cadr(x)) :
      eq(fn, 'ATOM') ? atom(car(x)) :
      eq(fn, 'EQ') ? eq(car(x), cadr(x)) :
      /* T ? */ lisp_apply(lisp_eval(fn, a), x, a)
    ) :
    eq(car(fn), 'LAMBDA') ?
      lisp_eval(caddr(fn), pairlis(cadr(fn), x, a)) :
    eq(car(fn), 'LABEL') ?
      lisp_apply(caddr(fn), x, cons(cons(cadr(fn), caddr(fn)), a)) :
    undefined

  console.log('apply <', pp(retval))
  return retval
}
// END: apply

// START: eval
lisp_eval = (e, a) => {
  console.log('eval =>', pp(e), pp(a))
  let retval =
    atom(e) ? cdr(assoc(e, a)) :
    atom(car(e)) ? (
      eq(car(e), 'QUOTE') ? cadr(e) :
      eq(car(e), 'COND') ? evcon(cdr(e), a) :
      /* T ? */ lisp_apply(car(e), evlis(cdr(e), a), a)
    ) :
    /* T ? */ lisp_apply(car(e), evlis(cdr(e), a), a)
  console.log('eval <=', pp(retval))
  return retval
}

cadar = l => car(cdr(car(l)))

evcon = (c, a) => {
  console.log('evcon ==>', pp(c), pp(a))
  let retval = lisp_eval(caar(c), a) ? lisp_eval(cadar(c), a) :
  /* T ? */ evcon(cdr(c), a)
  console.log('evcon <==', pp(retval))
  return retval
}
evlis = (m, a) => {
  console.log('evlis ==>', pp(m), pp(a))
  let retval = nil(m) ? 'NIL' : cons(lisp_eval(car(m), a), evlis(cdr(m), a))
  console.log('evlis <==', pp(retval))
  return retval
}
// END: eval

// START: evalquote
evalquote = (fn, x) => lisp_apply(fn, x, null)
// END: evalquote

// In case we want to start from a sexp-as-a-string, a simple parser
// START: parser
P = require('parsimmon')

parser = P.createLanguage({

  atom: () => P.regexp(/[A-Z][A-Z0-9]*/).desc('atom'),

  sexp: (r) => P.seqMap(
    P.string("("),
    r.sexp,
    P.string("."),
    r.sexp,
    P.string(")"),
    (_lpar, l, _dot, r, _rpar) => cons(l, r))
                .or(P.seqMap(
                  P.string("("),
                  P.sepBy(r.sexp, P.string(" ")),
                  P.string(")"),
                  (_lpar, elems, _rpar) => list(...elems)))
                .or(r.atom)
})

ps = (s) => parser.sexp.tryParse(s)

// pretty printer. A bit quick and dirty.
pp_cons_or_list = sexp => sexp == null ? '\b' : atom(sexp) ? ' . ' + sexp : pp_list(sexp, true)
pp_list = (sexp, in_list) => (in_list ? '' : '(') + pp(sexp[0]) + ' ' + pp_cons_or_list(sexp[1]) + (in_list ? '' : ')')
do_pp = sexp => atom(sexp) ? `${sexp}` : pp_list(sexp, false)
pp = sexp => do_pp(sexp).replace(' \b', '')

console.log(pp(ps("(AA B C)")))
console.log(pp(ps("(AA.(B.C))")))
console.log(pp(ps("(A B (C.(D.E)))")))
console.log(pp(ps("((A.B) (C.D))")))

assertEqual(ps("(AA.(B.C))"),
            ['AA', [ 'B', 'C']])
assertEqual(ps("ATOM"),
            'ATOM')
assertEqual(ps("(A B C)"),
            ['A', ['B', ['C', null]]])
// END: parser

assertEqual(evalquote('CAR', [['A', 'B']]), 'A')
// debug = true

exp = ps("(LAMBDA (X Y) (CONS (CAR X) Y))")
args = ps("((A B) (C D))")
answer = ps("(A C D)")
log_debug("exp", exp)
log_debug("args", args)

assertEqual(evalquote(exp, args), answer)

console.log('----------')
console.log('Passed ', testCount, 'tests')

// For reuse
module.exports = {
  assert,
  assertEqual
}
