/////////////////////////////////////////////////////////////////////////////
//
// This file implements a more "serious" Lisp machine, where we do
// things like garbage collection ourselves.
//

lodash = require('lodash')
util = require('util')
debug = false
log_debug = (...args) => debug && console.log(...args)

console.log("===========", new Date())

// simple helper for assertions.
testCount = 0
function assert(a) {
  testCount++
  if (!a)
    throw new Error("Assertion failed")
}
function assertEqual(a, b) {
  testCount++
  if (!lodash.isEqual(a, b))
    throw new Error(`Assertion failed, not equals: ${util.inspect(a)} != ${util.inspect(b)}`)
}

// We need a machine that can hold 32k words.
const memory_size = 32000
var memory = new Array(memory_size)
memory.fill(undefined)
Object.seal(memory)

// GC Roots. We implement this as a series of functions in case
// we have variable gc roots (which makes our life a bit easier)
var roots = []

memdump = function(n) {
  for (i = 0; i < n; i++) {
    console.log("Memory ", i, ':', memory[i])
  }
}

// A word is an object with three elements: car, cdr, flags
// accessors will work simplest.

car = function(loc) {
  m = memory[loc]
  return m == undefined ? undefined : m.car
}
cdr = function(loc) {
  m = memory[loc]
  return m == undefined ? undefined : m.cdr
}

FLAG_MARK = 0x01 // Mark in use for garbage collector
MASK_MARK = 0xff ^ FLAG_MARK
flags = function(loc) {
  m = memory[loc]
  return m == undefined ? undefined : m.flags
}
flag_mark = function(loc) {
  m =  memory[loc]
  if (m != undefined)
    memory[loc].flags = m.flags | FLAG_MARK
}
flag_unmark = function(loc) {
  m = memory[loc]
  if (m != undefined)
    memory[loc].flags = m.flags & MASK_MARK
}
has_mark = function(loc) {
  m = memory[loc]
  if (m != undefined)
    return m.flags & FLAG_MARK
}


// Set up the free list. We store the head of the free list at
// location two. Location zero has nil, one has true.
const free_list_loc = 2
const lowest_memory_loc = free_list_loc + 1
init_mem = function() {
  for (i = 2; i < memory_size; i++) {
    memory[i] = {
      car: null,
      cdr: i == memory_size - 1 ? null : i + 1,
      flags: 0
    }
  }

  memory[0] = null // we initialize NIL later on
  memory[1] = null // same for T
  memory[free_list_loc].cdr = free_list_loc + 1
}

init_mem()

console.log("Memory 0", memory[0])
console.log("Memory 1", memory[1])
console.log("Memory 2", memory[2])
console.log("Memory 3", memory[3])
console.log("Memory 4", memory[4])
console.log("Memory 5", memory[5])
console.log(" ...")
console.log("Memory 31998", memory[31998])
console.log("Memory 31999", memory[31999])

mark = function() {
  // Starting at each root, we mark all reachable memory locations.
  roots.forEach((root) => {
    console.log("== Start GC root", root())
    do_mark(root())
  })
}

do_mark = function(loc) {
  console.log("Marking from loc ", loc)
  if (has_mark(loc)) {
    console.log(" already marked, break:", loc)
    return; // already marked, we're done
  }
  if (loc < 90)
    console.log(" mark", loc)
  flag_mark(loc)

  maybe_mark(car(loc))
  maybe_mark(cdr(loc))
}

maybe_mark = function(loc) {
  if (Number.isInteger(loc) && loc >= lowest_memory_loc && loc < memory_size)
    do_mark(loc)
  else
    console.log("  bail on ", loc)
}

sweep = function() {
  // Rebuild the free list, resetting our markers as we go
  memory[free_list_loc].cdr = 0
  for (loc = lowest_memory_loc; loc < memory_size; loc++) {
    if (has_mark(loc)) {
      // marked means in use, all we need to do is unmark it again
      flag_unmark(loc)
    }
    else {
      // was not marked, so we add it to the free list
      memory[loc].car = null
      memory[loc].cdr = memory[free_list_loc].cdr
      memory[loc].flags = 0
      memory[free_list_loc].cdr = loc
    }
  }
}

garbage_collect = function() {
  mark()
  sweep()
  if (memory[free_list_loc].cdr == null) {
    throw "Out of memory"
  }
}

// allocate a cons cell. We return the location
// of the cell allocated
cons = function(a, b) {
  if (memory[free_list_loc].cdr == null) {
    garbage_collect()
  }
  // Allocate the first element on the list,
  // update the free list loc, clean out the word
  // allocated
  loc = memory[free_list_loc].cdr
  memory[free_list_loc].cdr = memory[loc].cdr
  memory[loc].car = a
  memory[loc].cdr = b
  memory[loc].flags = 0
  return loc
}

cons('A', 'B')
cons('C', 'D')
console.log("Memory 0", memory[0])
console.log("Memory 1", memory[1])
console.log("Memory 2", memory[2])
console.log("Memory 3", memory[3])
console.log("Memory 4", memory[4])
console.log("Memory 5", memory[5])
console.log(" ...")
console.log("Memory 31998", memory[31998])
console.log("Memory 31999", memory[31999])

// Enough demo, init again
init_mem()

// Atoms and plists

// NOTE
// In Lisp 1, these are called association lists with a footnote that
// in MIT Jargon, they're property lists. We follow modern usage, where
// an association list is of the form ((a . b) (c . d)) and a property
// list is ((a b) (c d)) and mostly used for attaching properties
// to symbols.

// These two deserve their own constants.
const NIL = 0
const T = 1

// first element of a property list has a flag to indicate it's such a list
const plist_flag = -1
// property list types, negative so they don't overlap with valid memory addresses
const plist_PNAME = -2 // Printable name, both for lookup and for printing
const plist_APVAL = -3 // Value when atom is a constant
//const plist_EXPR  = -4 // S-exp when atom is a function
const plist_SUBR  = -5 // Pointer to native code when atom is a native method

// Atoms have property lists. They're special, in that they
// start with a word that has "-1" in its car to signal a
// property list so that the rest of the elements in the
// list can have special treatment. The rest of the elements
// is alternating: plist type, plist value (always a pointer
// to the actual value to keep things regular whether values are
// immediate or not).
// Note: plists look like regular association lists (see `pairlis`) but
// the aren't. The -1 marker, for starters, but plists also can have flags that
// don't follow the key/value format.
is_plist = function(loc) {
  return memory[loc].car == plist_flag
}

atom = function(loc) {
  // Quick and dirty: short-circuit for NIL and T, else
  // if it has a plist, it's an atom.
  return loc == NIL || loc == T || caar(loc) == plist_flag
}

// Collect all them helpers here as we need them.
caar = l => car(car(l))
cadr = l => car(cdr(l))
caadr = l => car(car(cdr(l)))
cdar = l => cdr(car(l))
cddr = l => cdr(cdr(l))

get = function(loc, kind) {
  if (!atom(loc))
    return NIL

  plist_elm = cdar(loc)
  while (plist_elm) {
     if (cdr(plist_elm) == NIL)
       return NIL

     if (car(plist_elm) == kind) {
       return caadr(plist_elm)
     }

     plist_elm = cddr(plist_elm)
  }
  return NIL
}

plist_pname = function(loc) {
  return get(loc, plist_PNAME)
}
plist_apval = function(loc) {
  return get(loc, plist_APVAL)
}

// Predefined atoms. We need to do two things for an atom:
// 1. Allocate the plist and set it up
// 2. Insert it into the list of atoms

// NIL is special - we don't allocate it, it resides hardcoded in memory slot 0
// Therefore its value (APVAL) is 0, which is itself.
memory[NIL] = function() {
  plist = cons(plist_flag, cons(plist_APVAL,
                                cons(cons(NIL, 0),
                                     cons(plist_PNAME,
                                          cons(cons('NIL', 0), 0)))))
  return {car: plist, cdr: 0, flags: 0}
}()
// T as well
memory[T] = function() {
  plist = cons(plist_flag, cons(plist_APVAL,
                                cons(cons(T, 0),
                                     cons(plist_PNAME,
                                          cons(cons('T', 0), 0)))))
  return {car: plist, cdr: 0, flags: 0}
}()

// (The locations of) NIL and T are GC roots
roots.push(() => NIL)
roots.push(() => T)

assert(atom(NIL))
assertEqual(plist_pname(NIL), 'NIL')
assertEqual(plist_apval(NIL), NIL)
assert(atom(T))
assertEqual(plist_pname(T), 'T')
assertEqual(plist_apval(T), T)

// Atoms list. Empty list to start with.
var atoms = cons(0, 0)
console.log("Atoms cons is ", atoms)
roots.push(() => atoms) // atoms is a GC root

// Internally, atoms are just pointers so we need
// to look them up by name.
atom_by_name = function(name) {
  if (name == 'NIL')
    return NIL
  if (name == 'T')
    return T

  cur = atoms
  while (cur) {
    cur_name = plist_pname(cur)
    if (name == cur_name)
      return cur

    cur = cdr(cur)
  }
  return undefined
}

// define built-in functions.
defnative = function (name, js_func) {
  plist = cons(plist_flag, cons(plist_SUBR,
                                cons(cons(js_func, 0),
                                    cons(plist_PNAME,
                                         cons(cons(name, 0), 0)))))
  atoms = cons(plist, atoms)
}

defnative('CAR', car)
defnative('CDR', cdr)
defnative('ATOM', atom)
defnative('CADR', cadr)
defnative('CAADR', caadr)
defnative('CDAR', cdar)
defnative('CDDR', cddr)
defnative('GET', get)
console.log("Atoms start:", atoms)

memdump(10)
mark()

memdump(90)
sweep()
memdump(90)

assertEqual(plist_pname(atom_by_name('CAR')), 'CAR')
assertEqual(get(atom_by_name('CAR'), plist_SUBR), car)

// work up from here to eval/apply.

// garbage collection

console.log('----------')
console.log('Verified ', testCount, 'assertions')
